# Program    : CSV2HTML.AWK
# Purpose    : Convert a CSV file into an HTML table
# Date       : 18 February 2005
# Author     : Bob Jonkman <bjonkman@sobac.com>

# Copyright 2008 Bob Jonkman and/or SOBAC Microcomputer Services

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.


# Modified   : 3 February 2006 - Added navaids "Go to the top of this page"
#              31 March 2006   - Changed default from HTTP to NOHTTP primarily in dynamic pages
#                              - Added other code (eg. QUERY_STRING) for dynamic pages
#              25 Sept 2006    - Added "Prev, Next" navigational aids
#              9 January 2007  - Added id attributes to all TR elements
#              30 May 2007     - Using tempfile to place tfoot before tbody
#              6 March 2008    - Added table sorting with sorttable.js
#                                from http://www.kryogenix.org/code/browser/  by Stuart Langridge
#              21 May 2008     - Fixed bug where an empty <tbody> would come from previous file
#                                    (required considerable re-organizing of code)
#                              - Added "1 file(s) processed" logic

# Usage      : gawk -f CSV2HTML.AWK -f LIBRARY.AWK [-v TITLE=title] [0*(-v variable=(0|1) )] 1*(filename.csv)
#               http:\\hostname\csv2html.awk?FILENAME=filename.csv[1*(+filename.csv)][&TITLE=title][0*(&variable(0|1))]
# Note       : Use a literal + to separate filenames in URI

# Variables  :   TITLE = (text) HTML <title> and <h1> elements for the page
#                PLOT  = if TRUE Places FILENAME.png image element on right
#                LINK  = if TRUE Creates link to FILENAME for each table
#                NOHEADER = if TRUE Does not place the first line of CSV file in THEAD entity
#                NOHTTP  = if TRUE Suppresses a "Content-type: text/html" line for dynamic pages
#                SORTABLE = if TRUE adds script=sorttable.js to head and class="sortable" to tables

#
# Optional vars: FIELD_SEPARATOR  (default == ",")
#                FIELD_DELIMITER  (default == QUOTE )
#                RECORD_SEPARATOR (default == "\n")


function printbody(printtext)   {           # Send <tbody> parts to temporary file
            if(!tfootflag)
                print(printtext) > tempfile ;
            else
                print(printtext) ;
}

function finishtable()  {
                            if(tfootflag)
                            {   print("    </tfoot>") ;
                            }

                            if(tbodyflag)
                            {   
                                close(tempfile) ;   #   stop writing to tempfile
                                print("    <tbody>") ;

                                while(getvalue = (getline templine < tempfile) > 0)
                                {
                                    print(templine) ;     #   copy all records to standard output
                                }

                                close(tempfile) ;   #   stop reading from file

                                if(ENVIRON["windir"])
                                    system ("del " tempfile) ;  #   Clean up after yourself.
                                else
                                    system ("rm "  tempfile) ;  

                                print("    </tbody>") ;
                            } else  
                            {
                                if(tableflag)   # If a <table> was printed
                                {               #    then print a vestigial <tbody>
                                    print("    <tbody>") ;
                                    print("     <tr>") ;
                                    print("      <td>") ;
                                    print("      </td>") ;
                                    print("     </tr>") ;
                                    print("    </tbody>") ;
                                }
                            }

                            if(tableflag)
                            {
                                print("   </table>") 
                                print("   </div> <!-- end of id=" QUOTE valnameid(FILENAME) QUOTE " -->") ;
                                print("") 
                            }
                            tfootflag = 0 ;
                            tbodyflag = 0 ;
                            tableflag = 0 ;
                }


function makemenu(topofpage)    {
    # Create header/footer menu
    # parameter==TRUE then include "Top of page" link
            print("  <div class=" QUOTE "menu" QUOTE ) ;
            if(!topofpage)  # if this is NOT at the top of page then ...
                print("        id=" QUOTE "toc" QUOTE ) ;
            print("   >") 
            print("   <p>In this page:</p>") ;
            print("   <ul>" )

            if (topofpage)
            {
                print("    <li>[<a href=" QUOTE "#top-of-page" QUOTE " class=" QUOTE "internal" QUOTE ">Top of this page</a>] </li>")
                print("    <li>[<a href=" QUOTE "#toc" QUOTE " class=" QUOTE "internal" QUOTE ">Table of Contents</a>] </li>")
            }

            for(i=1; i<ARGC; i++)
            {
                print("    <li>[<a class=" QUOTE "internal" QUOTE " href=" QUOTE "#" valnameid(ARGV[i]) QUOTE ">" txt2html(ARGV[i]) "</a>] </li>") 
            }
            print("   </ul>")
            print("  </div> <!-- end of menu  -->")
}




BEGIN    {  version = "\nVeRsIoN= csv2html.awk 2008-05-22 \nCoPyRiGhT= Copyright under GNU GPLv3 2008 by Bob Jonkman  bjonkman@sobac.com \n"

            QUOTE = "\"" ;

            if(!FIELD_SEPARATOR)
                FIELD_SEPARATOR = "," ;

            if(!FIELD_DELIMITER)
                FIELD_DELIMITER = "\"" ;

            if(!RECORD_SEPARATOR)
                RECORD_SEPARATOR = "\n" ;

            tempfile = ENVIRON["TEMP"] "\\csv2html" PROCINFO["pid"] ".tmp" ;     # Temporary file for storing <tbody> parts

            parsecgi(ENVIRON["QUERY_STRING"],qstring)

            if(ARGC == 1)   # no files on the command line
            {
                if(qstring["FILENAME"])
                {
                    ARGC = 1 + split(qstring["FILENAME"],ARGV," ")
                }
            }

            if(!NOHTTP)
                NOHTTP = !(ENVIRON["REQUEST_METHOD"]) ;
            
            if(!TITLE)
                TITLE = qstring["TITLE"]

            if(!TITLE)          # if there is still no title then...
                TITLE = "Tables from CSV files"

            if(!PLOT)
                PLOT = qstring["PLOT"]

            if(!LINK)
                LINK = qstring["LINK"]

            if(!NOHEADER)
                NOHEADER = qstring["NOHEADER"]

            th = NOHEADER ? "td" : "th" ;  # if NOHEADER then first column cell is <td> else <th>

            if(!SORTABLE)
                SORTABLE = qstring["SORTABLE"]

            if(SORTABLE)
                LINKREL["sortable"] = "<script src=" QUOTE "/bin/javascript/sorttable.js" QUOTE " type=" QUOTE "text/javascript" QUOTE "></script>  <!-- from http://www.kryogenix.org/code/browser/  by Stuart Langridge -->" ;

        printhtmlhead(TITLE)

        print(" <body>") ;
        print("  <div>") ;
        print("   <a name=" QUOTE "top-of-page" QUOTE " id=" QUOTE "top-of-page" QUOTE ">");
        print("    <!-- " version " -->");
        print("   </a>") ;
        print("  </div>") ;

#        print("   <div id=" QUOTE "header" QUOTE ">") ;                        
#        print("    <a href=" QUOTE "/" QUOTE) ;                        
#        print("      title=" QUOTE "" QUOTE ">") ;
#        print("    <img") ;                        
#        print("        src=" QUOTE "/images/sitelogo.png" QUOTE ) ;                        
#        print("        alt=" QUOTE "Site logo" QUOTE ) ;                        
#        print("      style=" QUOTE "float: left ; height: 48px ; width: 150px ;" QUOTE " /></a>") ;
#
#        print("    <a href=" QUOTE "/" QUOTE "") ;                        
#        print("      title=" QUOTE "" QUOTE ">") ;
#        print("    <img") ;                        
#        print("        src=" QUOTE "/images/siteicon.png" QUOTE ) ;                        
#        print("        alt=" QUOTE "Site icon" QUOTE ) ;                        
#        print("      style=" QUOTE "float: right ; height: 48px ; width: 48px ;" QUOTE " /></a>") ;
#
#        print("    <br />") ;
#        print("    <hr />") ;
#        print("   </div><!-- end of header -->") ;
#        print("") ;

        print("  <h1>" makehtml(TITLE) "</h1>")

        if (ARGC == 1)
        {   print("  <div id=" QUOTE "content" QUOTE ">")
            print("   <p>No files specified, nothing to do!</p>")

            exit    # this will process END{}
        }

        if (ARGC > 2)       # If there is more than one file on the page
            makemenu(0) ;   #   create header menu without "Top of page" link

        print("  <div id=" QUOTE "content" QUOTE ">")

} # end of BEGIN


(FNR == 1)  {   # Finish off the previous file
                if(ARGIND > 1)        # check if we've processed multiple files
                    finishtable() ;

        
        print("   <div  id=" QUOTE valnameid(FILENAME) QUOTE ">") ;
        print("   <h2")
        if (ARGIND > 1)   #  Suppress page-break-before on the first file
            print("        style=" QUOTE "page-break-before: always ; " QUOTE )

        print("       >")

        if (ARGC > 2)   # Display "Prev, Top, Next" navigational aids
        {
            print("    <span class=" QUOTE "internal navaid" QUOTE ">")

            if (ARGIND > 1)       # suppress "Previous" for first file    
            {
                print("     <a href=" QUOTE "#" valnameid(ARGV[ARGIND-1]) QUOTE)
                print("        title=" QUOTE "Previous Table: " txt2html(ARGV[ARGIND-1]) QUOTE)
                print("     >&larr;</a>")
            } else
            {   print("     <span class=" QUOTE "disabled" QUOTE ">&larr;</span>")
            }

            print("     <a href=" QUOTE "#top-of-page" QUOTE)
            print("        title=" QUOTE "Go to the top of this page" QUOTE)
            print("     >&thinsp;&uarr;&thinsp;</a>")

            if (ARGIND < ARGC-1 )    # suppress "Next" for last file
            {
                print("     <a href=" QUOTE "#" valnameid(ARGV[ARGIND+1]) QUOTE)
                print("        title=" QUOTE "Next Table: " txt2html(ARGV[ARGIND+1]) QUOTE)
                print("     >&rarr;</a>")
            } else
            {   print("     <span class=" QUOTE "disabled" QUOTE ">&rarr;</span>")
            }
        print("    </span>")
        }

        print("   " txt2html(FILENAME) "</h2>" )


        print("   <p>Generated on <abbr class=" QUOTE "datetime" QUOTE " title=" QUOTE strftime("%Y-%m-%dT%H:%M:%S") QUOTE ">" strftime("%a, %d %b %Y %H:%M:%S %z") "</abbr> from ") 

        if(LINK)
            print("    <a href=" QUOTE txt2uri(FILENAME) QUOTE ">" txt2html(FILENAME) "</a>") 
        else
            print("    " txt2html(FILENAME))

        print("   </p>")
        print("") 
        if(PLOT)            
        {   print("   <img src=" QUOTE txt2uri(FILENAME) ".png" QUOTE )
            print("        alt=" QUOTE "Plot of " txt2html(FILENAME) QUOTE )
            print("        style=" QUOTE "float: right ; " QUOTE )
            print("        />") 
        }

        tableflag = 1 ;  # Yes, we've started a table
        print("   <table") 
        print("     border=" QUOTE "1" QUOTE )

        if (SORTABLE)
            print("      class=" QUOTE "sortable" QUOTE ) ;

        print("    summary=" QUOTE "Generated on " strftime("%a, %d %b %Y %H:%M:%S %z") " from " txt2html(FILENAME) QUOTE) 
        print("         >") 

} # end of (FNR == 1)

((FNR == 1) && (!NOHEADER))  {        # Do the headers, except if NOHEADER is specified
        print("    <thead>") 
        print("     <tr id=" QUOTE "line" FNR "file" ARGIND QUOTE ">") 
        delete hdr 
        
        numfields = parsecsv($0,header) 
    
        for(i=1; i <= numfields; i++)
        {
            hdr[i] = "hdr" i "file" ARGIND 
            print("      <th") 
            print("       id=" QUOTE hdr[i] QUOTE )      # doesn't need validnameid(hdr[i]) because we've generated hdr[i]
#           print("       style=" QUOTE "border: thin solid ;" QUOTE )
            print("         >" makehtml(header[i]) "</th>") 
        }

        print("     </tr>") 
        print("    </thead>") 

        bodyflag = 1   # reading tbody

        next 
} #end of ((FNR == 1) && (!NOHEADER))


($0 == "=====") {   # My convention for signalling footers in CSV files is "====="
                    tfootflag = 1      # now writing <tfoot>
                    print("    <tfoot>") ;
                    next               # Don't display the footer separator itself

} #end of ($0 == "=====")


((FNR > 1) || (NOHEADER))    {  
        if(!tfootflag)      # If we're not writing <tfoot>...
            tbodyflag = 1 ; # ... we must be writing <tbody>       
    
        numfields = parsecsv($0,row) 
                
        printbody("     <tr   id=" QUOTE "line" FNR "file" ARGIND QUOTE ) ;
        printbody("        class=" QUOTE (FNR % 2 ? "odd" : "even" ) QUOTE ">") ;
        printbody("      <" th)       # Note that the first element in a column could be replaced with <th></th>

        if (hdr[1])
            printbody("           headers=" QUOTE hdr[1] QUOTE )

        printbody("         >" makehtml(row[1]) "</" th ">")    

        for(i=2; i <= numfields; i++)
        {    printbody("      <td") 
            if (hdr[i])
               printbody("      headers=" QUOTE hdr[i] QUOTE ) 
            printbody("          >" makehtml(row[i]) "</td>") 
        }

        printbody("     </tr>") 

    } #end of ((FNR > 1) || (NOHEADER))


END {   if(ARGIND)    # Tables are ended only if files have been processed
        {    
            finishtable() ;
        }

        print("   <p>" ARGIND " file" ((ARGIND==1) ? "" : "s") " processed.</p>") 
        print("  </div> <!-- end of content -->") 

        if (ARGC > 2)       # If there is more than one file on the page
        {   print("  <div id=" QUOTE "footer" QUOTE ">")
            makemenu(1) ;   #   Create footer menu with "Top of page" link
            print("  </div> <!-- end of footer -->")
        }

        print(" </body>")     
        print("</html>")

    } #end of END

# EOF: CSV2HTML.AWK
